require("dotenv").config();
const db = require("../models");
const Sender = db.sender;
const User = db.user;
const Image = db.image;
const Op = db.Sequelize.Op;
const moment = require("moment");
const crypto = require("crypto");
const { v4: uuidv4 } = require("uuid");
const { off } = require("process");
const { paginate } = require("../helper/helper");

exports.create = async (req, res, next) => {
  try {
    let {
      user_id,
      packageName,
      packagePickupDestination,
      packageDestination,
      deliveryTime,
      packageLength,
      width,
      height,
      packageWeight
    } = req.body;
    console.log(req.body.images);
    // Validate user input
    if (
      !(
        user_id &&
        packageName &&
        packageDestination &&
        packagePickupDestination &&
        deliveryTime &&
        packageWeight &&
        packageLength &&
        width &&
        height
      )
    ) {
      return res
        .status(200)
        .send({ status: false, message: "All input is required", data: null });
    }

    if (
      typeof packageLength !== "number" &&
      typeof packageWeight !== "number" &&
      typeof width !== "number" &&
      typeof height !== "number"
    ) {
      return res.status(200).send({
        status: false,
        message:
          "Package Weight, Package Length, Width and Height need to be a number.",
        data: null
      });
    }

    const findUser = await User.findOne({
      where: { id: user_id }
    });

    if (!findUser) {
      return res.status(200).send({
        status: false,
        message: "User with that id not found",
        data: null
      });
    }

    if (deliveryTime < moment(Date.now()).format("YYYY-MM-DD")) {
      return res.status(200).send({
        status: false,
        message: "Date cant be smaller then present date.",
        data: null
      });
    }

    const verificationCode = crypto
      .createHash("sha1")
      .update(Date.now() + uuidv4())
      .digest("hex")
      .slice(0, 32);

    const sender = {
      user_id: req.body.user_id,
      packageName: packageName,
      packagePickupDestination: packagePickupDestination,
      packageDestination: packageDestination,
      deliveryTime: moment(new Date(deliveryTime)).format("YYYY-MM-DD"),
      packageLength: packageLength,
      width: width,
      height: height,
      packageWeight: packageWeight,
      verificationCode: verificationCode,
      images: req.body.images
    };

    Sender.create(sender, { include: [{ model: Image, as: "images" }] })
      .then(data => {
        return res.status(200).send({
          status: true,
          message: "Sender created successfuly.",
          data: data
        });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message:
            err.message || "Some error occurred while creating the sender.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.edit = async (req, res, next) => {
  // Our edit logic starts here
  try {
    const { id } = req.params;
    let {
      user_id,
      teleporter_id,
      packageName,
      packagePickupDestination,
      packageDestination,
      deliveryTime,
      packageLength,
      width,
      height,
      packageWeight,
      verificationCode,
      status
    } = req.body;

    const oldSender = await Sender.findOne({ where: { id: id } });

    if (!oldSender) {
      return res
        .status(200)
        .send({ status: false, message: "Sender not found.", data: null });
    }

    if (user_id) oldSender.user_id = user_id;
    if (teleporter_id) oldSender.teleporter_id = teleporter_id;
    if (packageName) oldSender.packageName = packageName;
    if (packageDestination) oldSender.packageDestination = packageDestination;
    if (packagePickupDestination)
      oldSender.packagePickupDestination = packagePickupDestination;
    if (deliveryTime)
      oldSender.deliveryTime = moment(new Date(deliveryTime)).format(
        "YYYY-MM-DD"
      );
    if (packageLength) oldSender.packageLength = packageLength;
    if (width) oldSender.width = width;
    if (height) oldSender.height = height;
    if (packageWeight) oldSender.packageWeight = packageWeight;
    if (verificationCode) oldSender.verificationCode = verificationCode;
    if (status) oldSender.status = status;

    Sender.update(req.body, {
      where: { id: id }
    })
      .then(async num => {
        const updated = await Sender.findOne({ where: { id: id } });

        return res
          .status(200)
          .send({ status: true, message: "Updated", data: updated });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Error updating record",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.getAll = async (req, res, next) => {
  // Our getAll logic starts here
  try {
    const allSenders = await Sender.findAll();

    Sender.findAll()
      .then(data => {
        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched.", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message:
            err.message || "Some error occurred while fetching the senders.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.delete = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { id } = req.params;

    const sender = await Sender.findOne({ where: { id: id } });

    if (!sender) {
      return res
        .status(200)
        .send({ status: false, message: "Sender not found", data: null });
    }
    Sender.destroy({ where: { id: id } })
      .then(data => {
        return res.status(200).send({
          status: true,
          message: "Successfuly deleted record.",
          data: sender
        });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Deleting faild.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.getById = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { id } = req.params;

    Sender.findOne({ where: { id: id } })
      .then(data => {
        if (!data) {
          return res
            .status(200)
            .send({ status: false, message: "Sender not found", data: null });
        }
        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched.", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
  } catch (err) {
    console.log(err);
  }
};

exports.search = async (req, res, next) => {
  try {
    const { user_id, page, limit } = req.params;
    let { searchTerm, from, to, departure } = req.body;

    if (departure < moment(Date.now()).format("YYYY-MM-DD")) {
      return res.status(200).send({
        status: false,
        message: "Date cant be smaller then present date.",
        data: null
      });
    }

    if (!(searchTerm || (from && to && departure))) {
      return res.status(200).send({
        status: false,
        message: "Some input is required.",
        data: null
      });
    }

    if (from && to && departure) {
      await Sender.findAndCountAll({
        where: [
          {
            packagePickupDestination: { [Op.like]: `%${from}%` },
            packageDestination: { [Op.like]: `%${to}%` },
            deliveryTime: { [Op.gte]: departure }
          },
          { user_id: { [Op.not]: user_id } }
        ],
        include: Image,
        distinct: true,
        order: [["createdAt", "DESC"]],
        ...paginate({ currentPage: page, pageSize: limit })
      })
        .then(data => {
          if (!data["rows"] || data["rows"].length == 0) {
            return res.status(200).send({
              status: false,
              message: "There is no available senders for this search!",
              data: data
            });
          }

          return res.status(200).send({
            status: true,
            message: "Successfuly fetched!",
            data: data
          });
        })
        .catch(err => {
          return res.status(200).send({
            status: false,
            message: err.message || "Faild.",
            data: null
          });
        });
      return next();
    }

    if (searchTerm) {
      await Sender.findAndCountAll({
        where: {
          [Op.or]: [
            { packageName: { [Op.like]: `%${searchTerm}%` } },
            { packageDestination: { [Op.like]: `%${searchTerm}%` } },
            { packagePickupDestination: { [Op.like]: `%${searchTerm}%` } }
          ],
          deliveryTime: { [Op.gte]: moment(Date.now()).format("YYYY-MM-DD") }
        },
        include: Image,
        distinct: true,
        order: [["createdAt", "DESC"]],
        ...paginate({ currentPage: page, pageSize: limit })
      })
        .then(data => {
          if (!data["rows"] || data["rows"].length == 0) {
            return res.status(200).send({
              status: false,
              message: "There is no available senders for this search!",
              data: null
            });
          }
          return res.status(200).send({
            status: true,
            message: "Successfuly fetched!",
            data: data
          });
        })
        .catch(err => {
          return res.status(200).send({
            status: false,
            message: err.message || "Faild.",
            data: null
          });
        });
      return next();
    }
  } catch (err) {
    console.log(err);
  }
};

exports.activePackages = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { userId, page, limit } = req.params;

    if (!userId) {
      return res
        .status(200)
        .send({ status: false, message: "Missing userId.", data: null });
    }

    await Sender.findAndCountAll({
      where: {
        [Op.or]: [{ status: "Active" }, { status: "Teleporting" }],
        user_id: userId
      },
      include: Image,
      distinct: true,
      order: [["createdAt", "DESC"]],
      ...paginate({ currentPage: page, pageSize: limit })
    })
      .then(data => {
        if (!data["rows"] || data["rows"].length == 0) {
          return res.status(200).send({
            status: false,
            message: "There is no available senders!",
            data: null
          });
        }

        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched!", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
    return next();
  } catch (err) {
    console.log(err);
  }
};

exports.historyPackages = async (req, res, next) => {
  // Our login logic starts here
  try {
    const { userId, page, limit } = req.params;

    if (!userId) {
      return res
        .status(200)
        .send({ status: false, message: "Missing userId.", data: null });
    }

    await Sender.findAndCountAll({
      where: {
        [Op.or]: [{ status: "Complited" }, { status: "Expired" }],
        user_id: userId
      },
      include: Image,
      distinct: true,
      order: [["createdAt", "DESC"]],
      ...paginate({ currentPage: page, pageSize: limit })
    })
      .then(data => {
        if (!data["rows"] || data["rows"].length == 0) {
          return res.status(200).send({
            status: false,
            message: "There is no available history!",
            data: null
          });
        }

        return res
          .status(200)
          .send({ status: true, message: "Successfuly fetched!", data: data });
      })
      .catch(err => {
        return res.status(200).send({
          status: false,
          message: err.message || "Faild.",
          data: null
        });
      });
    return next();
  } catch (err) {
    console.log(err);
  }
};
